using System;
using System.Data;
using System.Collections;
using System.Reflection;
using System.Diagnostics;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{

		#region Header

    ///
    ///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
    ///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
    ///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
    ///<Developers>
    ///	<Developer>John Vrooland</Developer>
    ///</Developers>
    ///<SiteName>Hines OIFO</SiteName>
    ///<CreationDate>1/13/2003</CreationDate>
    ///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
    ///<summary>VbecsProcess business object class</summary>

		#endregion

    public class VbecsProcess : BaseBusinessObject
    {
        /// <summary>
        /// Minimum columns in table
        /// </summary>
        public const string MINCOLUMNSINTABLE = "VbecsProcessId,VbecsProcessName";

		private DataRow _vbecsProcessRow;
        private DataTable _vbecsWorkLoad;

        private static DataSet _vistALMIP;

        private DataView _workLoads; 
        private DataTable _workLoadSave;

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1587"> 
        ///		<ExpectedInput>Valid data row</ExpectedInput>
        ///		<ExpectedOutput>Valid VbecsProcess object</ExpectedOutput>
        ///	</Case>
         ///
        ///<Case type="1" testid ="2540"> 
        ///		<ExpectedInput>Invalid data row</ExpectedInput>
        ///		<ExpectedOutput>Exception</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// VbecsProcess constructor with datarow parameter
        /// </summary>
        /// <param name="dr"></param>
        public VbecsProcess( DataRow dr ) : base()
        {
            _vbecsProcessRow = dr;
            this.IsDirty = false;

            if( _vbecsWorkLoad == null ) 
            {
                _vbecsWorkLoad = DAL.WorkLoad.GetVbecsWorkLoadForDivision( DivisionCode ); 
                DataColumn[] dcP =  { _vbecsWorkLoad.Columns[TABLE.WorkloadProcess.WorkloadProcessGuid] };
                _vbecsWorkLoad.PrimaryKey = dcP;
                _workLoadSave = _vbecsWorkLoad.Clone();
                _vbecsWorkLoad.AcceptChanges();
            }

            _workLoads = new DataView( _vbecsWorkLoad );
            _workLoads.RowFilter = "VbecsProcessId = " + this.VbecsProcessId.ToString();
            _workLoads.Sort = TABLE.WorkloadProcess.WorkloadCode;
        }

		static DataSet VistALMIP
        {
            get 
            {
                if( _vistALMIP == null )
                    _vistALMIP = gov.va.med.vbecs.DAL.VAL.WorkloadCodes.GetWorkloadCodeData();

                return _vistALMIP;
            }
        }

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1372"> 
		///		<ExpectedInput>valid code</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1373"> 
		///		<ExpectedInput>Invalid code</ExpectedInput>
		///		<ExpectedOutput>Empty String</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetWeightMultiplier
		/// </summary>
		/// <param name="workloadCode"></param>
		/// <returns></returns>
		public static string GetWeightMultiplier(string workloadCode)
		{		
			DataRow [] dtRow = VistALMIP.Tables[0].Select("LMIP = '" + workloadCode + "'");
			
			if (dtRow.Length > 0)
				return dtRow[0][TABLE.WorkloadProcess.WeightMultiplier].ToString();
			else
				return string.Empty;
		}

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/10/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2541"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>Valid boolean</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1374"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// Returns an array of all VBECS processes
        /// </summary>
        /// <returns>DataTable</returns>
        public bool CheckDirty()
        {
            return _workLoadSave.Rows.Count > 0;
        }

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/10/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1360"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>VbecsProcess array</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1375"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// Returns an array of all VBECS processes
        /// </summary>
        /// <returns>DataTable</returns>
        public static VbecsProcess[] GetAllVbecsProcesses()
        {
            DataTable _dt = DAL.WorkLoad.GetAllVbecsProcesses();
            ArrayList _vpAr = new ArrayList( _dt.Rows.Count );

            foreach(DataRow _dr in _dt.Rows)
                _vpAr.Add( new VbecsProcess( _dr ) );

            return (VbecsProcess[])_vpAr.ToArray( typeof(VbecsProcess) );
        }


        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1588"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>Valid data table</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1386"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Returns all Vista workloads
        /// </summary>
        /// <returns></returns>
        public static DataTable GetAllVistaWorkloads()
        {
            if( !VistALMIP.Tables.Contains( "Code" ) )
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					Common.Utility.ParseStringwithSpaces( MethodBase.GetCurrentMethod().Name ) ).ResString );	
			
			return VistALMIP.Tables["Code"];
        }

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/21/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1361"> 
        ///		<ExpectedInput>Valid workload code string</ExpectedInput>
        ///		<ExpectedOutput>Valid workload object</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="2542"> 
        ///		<ExpectedInput>invalid Valid workload code string</ExpectedInput>
        ///		<ExpectedOutput>ArgumentException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Returns a new workload object based on a workload code parameter
        /// </summary>
        /// <param name="workLoadCode"></param>
        /// <returns></returns>
        public WorkLoad GetNewWorkLoad(string workLoadCode)
        {
            if( workLoadCode == null || workLoadCode.Length > 10 )
                throw new ArgumentException( "Invalid Argument:", "workLoadCode" );

            try
            {
                WorkLoad workLoad = null;
                DataRow workLoadRow = null;
                DataRow[] cptCodes = null;

                DataRow[] vistaWorkLoadRows = VistALMIP.Tables[0].Select("LMIP = '" + workLoadCode + "'");
                string procedure = "";
                decimal cost = 0.0m;
				byte weightMultiplier = 0;

                if (vistaWorkLoadRows.Length > 0) 
                {
                    procedure = vistaWorkLoadRows[0][ARTIFICIAL.Procedure].ToString();
                    cost = vistaWorkLoadRows[0][ARTIFICIAL.Cost].ToString() == ""  ? 0.0m : decimal.Parse(vistaWorkLoadRows[0][ARTIFICIAL.Cost].ToString());
                    cptCodes = vistaWorkLoadRows[0].GetChildRows("Code_CPTCode");
					//TODO: VistA data coming across as decimal; defined in VBECS as tinyint - What is right?
					try
					{
						weightMultiplier = vistaWorkLoadRows[0][TABLE.WorkloadEvent.WeightMultiplier].ToString() == string.Empty ? byte.MinValue : byte.Parse( vistaWorkLoadRows[0][TABLE.WorkloadEvent.WeightMultiplier].ToString() );
					}
					catch{};
                }
				//
                workLoadRow = _vbecsWorkLoad.NewRow();
                workLoadRow[TABLE.WorkloadProcess.WorkloadProcessGuid] = Guid.NewGuid();
                // workLoadRow[ARTIFICIAL.Procedure] = procedure;
				workLoadRow["WorkloadProcedure"] = procedure;
                workLoadRow[ARTIFICIAL.Cost] = cost;
				workLoadRow[TABLE.WorkloadProcess.WeightMultiplier] = weightMultiplier;
                workLoadRow[TABLE.WorkloadProcess.WorkloadCode] = workLoadCode;
                workLoadRow[TABLE.WorkloadProcess.DivisionCode] = DivisionCode;
                workLoadRow[TABLE.WorkloadProcess.LastUpdateDate] = VBECSDateTime.GetDivisionCurrentDateTime();
                workLoadRow[TABLE.WorkloadProcess.LastUpdateUser] = Common.LogonUser.LogonUserName;
                workLoadRow[TABLE.WorkloadProcess.VbecsProcessId] = this.VbecsProcessId;
				workLoadRow[TABLE.WorkloadProcess.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
                _vbecsWorkLoad.Rows.Add(workLoadRow);
				//
                if( cptCodes == null || cptCodes.Length == 0 ) 
                    workLoad = new WorkLoad(workLoadRow);
                else
                    workLoad = new WorkLoad(workLoadRow, cptCodes);

                return workLoad;
            }
            catch( Exception ex )
            {
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString, ex);	
			}
        }

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1589"> 
        ///		<ExpectedInput>Valid Guid</ExpectedInput>
        ///		<ExpectedOutput>Valid workload</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="2543"> 
        ///		<ExpectedInput>Invalid Guid</ExpectedInput>
        ///		<ExpectedOutput>null</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Returns workload based on a workload GUID
        /// </summary>
        /// <param name="workLoadGuid"></param>
        /// <returns></returns>
        public WorkLoad GetWorkLoad( System.Guid workLoadGuid )
        {
            try
            {
                WorkLoad workLoad = null;
                DataRow workLoadRow = null;
                DataRow[] cptCodes = null;
                string workLoadCode;
        
                DataRow[] workLoadRows = _vbecsWorkLoad.Select("WorkLoadProcessGuid = '" + workLoadGuid + "'");
        
				if( workLoadRows.Length <= 0 )
					return null;

                workLoadRow = workLoadRows[0];
                workLoadCode = workLoadRow[TABLE.WorkloadProcess.WorkloadCode].ToString();

				DataRow[] vistaWorkLoadRows = VistALMIP.Tables[0].Select("LMIP = '" + workLoadCode + "'");
                string procedure = "";
                decimal cost = 0.0m;
				int weightMultiplier = 0;

                if (vistaWorkLoadRows.Length > 0) 
                {
                    procedure = vistaWorkLoadRows[0][ARTIFICIAL.Procedure].ToString();
                    cost = ((string)vistaWorkLoadRows[0][ARTIFICIAL.Cost] == string.Empty)  ? 0.0m : decimal.Parse(vistaWorkLoadRows[0][ARTIFICIAL.Cost].ToString());
                    cptCodes = vistaWorkLoadRows[0].GetChildRows("Code_CPTCode");
					//TODO: Bad VistA data?
					//HACK: remove try-catch when resolved.
					try
					{
						weightMultiplier = ( (string) vistaWorkLoadRows[0][TABLE.WorkloadProcess.WeightMultiplier]  == string.Empty) ? 0 : int.Parse(vistaWorkLoadRows[0][TABLE.WorkloadEvent.WeightMultiplier].ToString() );
					}
					catch{}
                }

                // workLoadRow[ARTIFICIAL.Procedure] = procedure;
				workLoadRow["WorkloadProcedure"] = procedure;
                workLoadRow[ARTIFICIAL.Cost] = cost;
				workLoadRow[TABLE.WorkloadEvent.WeightMultiplier] = weightMultiplier;
                
                if (cptCodes == null || cptCodes.Length == 0) 
                    workLoad = new WorkLoad(workLoadRow);
                else
                    workLoad = new WorkLoad(workLoadRow, cptCodes);

                return workLoad;
            }
            catch( Exception ex )
            {
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString, ex);	
			}
        }

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/21/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1356"> 
        ///		<ExpectedInput>Valid data row</ExpectedInput>
        ///		<ExpectedOutput>Valid data row</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1387"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        ///  Unused. Loads a data row from an object.
        /// </summary>
        public override DataRow LoadDataRowFromThis( DataRow dr )
        {
            return dr;
        }

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/14/2003</CreationDate>
        /// <summary>
        /// Unused. Loads an object from a data row.
        /// </summary>
        protected override void LoadFromDataRow(DataRow dr)
        {
        }

	

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/6/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2544"> 
        ///		<ExpectedInput>UpdateFunction</ExpectedInput>
        ///		<ExpectedOutput>No exceptions</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="2545"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Saves VbecsProcess changes
        /// </summary>
        /// <param name="lastUpdateFunctionId">UC or calling method</param>
        public void SaveChanges( Common.UpdateFunction lastUpdateFunctionId )
        {
            if( _workLoadSave.Rows.Count > 0 ) 
            {
                DAL.WorkLoad.SaveWorkLoadChanges( _workLoadSave, lastUpdateFunctionId );
                _workLoadSave = null;
                _vbecsWorkLoad = null;
            }

            this.IsDirty = false;
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1590"> 
        ///		<ExpectedInput>Valid workload</ExpectedInput>
        ///		<ExpectedOutput>No exceptions</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="2546"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Saves workload item
        /// </summary>
        /// <param name="wl"></param>
        public void AddItem(WorkLoad wl)
        {
            wl.WorkLoadRow[TABLE.WorkloadProcess.LastUpdateUser] = Common.LogonUser.LogonUserName;
            wl.WorkLoadRow[TABLE.WorkloadProcess.VbecsProcessId] = this.VbecsProcessId;

            DataRow[] drExist = _workLoadSave.Select("WorkloadProcessGuid = '" + wl.WorkloadProcessGuid.ToString() + "'");
            
			if (drExist.Length > 0) 
            {
                // Address SchemaBinding action
                drExist[0].Table.Columns[TABLE.WorkloadProcess.RowVersion].ReadOnly = false;
                drExist[0].ItemArray = wl.WorkLoadRow.ItemArray;
            }
            else
            {
				// _workLoadSave is the DataTable used to store workloads
                _workLoadSave.ImportRow(wl.WorkLoadRow);
            }

            wl.WorkLoadRow.AcceptChanges();
        }

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3589"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3590"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECS process Guid
		/// </summary>
		public System.Guid VbecsProcessGuid
		{
			get 
			{
				return (System.Guid)_vbecsProcessRow[TABLE.VbecsProcess.VbecsProcessGuid];
			}
		}

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1591"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>Valid int</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1391"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// VBECS process ID
        /// </summary>
        public int VbecsProcessId
        {
            get 
            {
                return (int)_vbecsProcessRow[TABLE.WorkloadProcess.VbecsProcessId];
            }
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1592"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>Valid string</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1394"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// VBECS process name
        /// </summary>
        public string VbecsProcessName
        {
            get 
            {
                return _vbecsProcessRow[TABLE.VbecsProcess.VbecsProcessName].ToString();
            }
        }

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/4/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1593"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>DataView</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1395"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Workloads dataview
        /// </summary>
        public DataView WorkLoads
        {
            get 
            {
                return _workLoads;
            }
        }

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1396"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1458"> 
		///		<ExpectedInput>Invalid string</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// DivisionCode - Only debug version 
		/// </summary>
		public string DivisionCode
		{
			get 
			{
				return Common.LogonUser.LogonUserDivisionCode;
			}
			set 
			{
				Common.LogonUser.LogonUserDivisionCode = value;
			}
		}
    }
}
